/**
 * Referral Tracking JavaScript for Tenant Referral Program
 * 
 * Extracted and adapted from template-referral-signup-test.php
 * Handles cookie management, URL parameter extraction, and API tracking
 */

// IMMEDIATE TEST - This should appear first
console.log('🚀 TRP: JAVASCRIPT FILE IS LOADING RIGHT NOW!');

// Immediate debug log to test if file is executing
console.log('TRP: JavaScript file loaded and executing');

(function($) {
    'use strict';
    
    console.log('TRP: Inside jQuery wrapper, $ is:', typeof $);
    
    try {
        console.log('TRP: Starting script execution...');
        
        // Global variables
        let finalRefCode = null;
        let cookieName = 'tenant_ref_code';
        let cookieExpirationDays = 30;
        let trackApiUrl = null;
        
        console.log('TRP: Variables declared successfully');
    
    /**
     * Cookie helper functions
     */
    function setCookie(name, value, days) {
        let expires = "";
        if (days) {
            const date = new Date();
            date.setTime(date.getTime() + (days * 24 * 60 * 60 * 1000));
            expires = "; expires=" + date.toUTCString();
        }
        const encodedValue = encodeURIComponent(value || "");
        let cookieString = name + "=" + encodedValue + expires + "; path=/; SameSite=Lax";
        if (window.location.protocol === 'https:') { 
            cookieString += "; Secure"; 
        }
        document.cookie = cookieString;
        console.log('TRP: Cookie set:', name, '(encoded value: ' + encodedValue + ')', 'Expires in', days, 'days.');
    }

    function getCookie(name) {
        const nameEQ = name + "=";
        const ca = document.cookie.split(';');
        for (let i = 0; i < ca.length; i++) {
            let c = ca[i];
            while (c.charAt(0) === ' ') { 
                c = c.substring(1, c.length); 
            }
            if (c.indexOf(nameEQ) === 0) {
                const encodedValue = c.substring(nameEQ.length, c.length);
                try {
                    const decodedValue = decodeURIComponent(encodedValue);
                    console.log('TRP: Cookie found:', name, '(decoded value: ' + decodedValue + ')');
                    return decodedValue;
                } catch (e) {
                    console.error('TRP: Error decoding cookie:', name, encodedValue, e);
                    return encodedValue; 
                }
            }
        }
        console.log('TRP: Cookie not found:', name);
        return null;
    }
    
    /**
     * Update referral notice display
     */
    function updateMainNotice(message, type, noticeElement) {
        if (!noticeElement) return;
        
        noticeElement.innerHTML = message;
        let noticeStyle = noticeElement.style;
        
        // Reset styles
        noticeStyle.padding = '';
        noticeStyle.backgroundColor = '';
        noticeStyle.border = '';
        noticeStyle.borderRadius = '';
        
        switch(type) {
            case 'success':
                noticeStyle.color = 'var(--trp-success-text, #155724)';
                noticeStyle.padding = '10px';
                noticeStyle.backgroundColor = 'var(--trp-success-bg, #d4edda)';
                noticeStyle.border = '1px solid var(--trp-success-border, #c3e6cb)';
                noticeStyle.borderRadius = '4px';
                break;
            case 'info':
                noticeStyle.color = 'var(--trp-info-text, #0c5460)';
                noticeStyle.padding = '0';
                noticeStyle.backgroundColor = 'transparent';
                noticeStyle.border = 'none';
                break;
            case 'warning':
                noticeStyle.color = 'var(--trp-warning-text, #856404)';
                noticeStyle.padding = '10px';
                noticeStyle.backgroundColor = 'var(--trp-warning-bg, #fff3cd)';
                noticeStyle.border = '1px solid var(--trp-warning-border, #ffeeba)';
                noticeStyle.borderRadius = '4px';
                break;
            default: // error
                noticeStyle.color = 'var(--trp-error-text, #721c24)';
                noticeStyle.padding = '10px';
                noticeStyle.backgroundColor = 'var(--trp-error-bg, #f8d7da)';
                noticeStyle.border = '1px solid var(--trp-error-border, #f5c6cb)';
                noticeStyle.borderRadius = '4px';
        }
    }
    
    /**
     * Track landing page visit via API
     */
    function trackLandingPageVisit(referralCode) {
        if (!trackApiUrl || !referralCode) {
            console.log('TRP: Cannot track landing page visit - missing API URL or referral code');
            return;
        }
        
        const payload = { 
            referralCode: referralCode, 
            event_type: 'LANDING_PAGE_VISIT' 
        };
        
        const requestOptions = {
            method: 'POST',
            headers: { 'Content-Type': 'application/json' },
            body: JSON.stringify(payload)
        };
        
        // Add ngrok headers if needed
        if (trackApiUrl.includes('ngrok-free.app') || trackApiUrl.includes('ngrok.io')) {
            requestOptions.headers['ngrok-skip-browser-warning'] = 'true';
        }
        
        console.log('TRP: Tracking landing page visit for code:', referralCode);
        
        fetch(trackApiUrl, requestOptions)
            .then(response => {
                if (!response.ok) {
                    return response.json().catch(() => ({
                        error: `Server responded with ${response.status}`
                    })).then(body => {
                        throw new Error(`HTTP error ${response.status}: ${body.error || response.statusText}`);
                    });
                }
                return response.json();
            })
            .then(data => {
                console.log('TRP: LANDING_PAGE_VISIT tracked successfully.', data);
            })
            .catch(error => {
                console.error('TRP: Error tracking LANDING_PAGE_VISIT:', error);
            });
    }
    
    /**
     * Process referral code from URL or cookie
     */
    function processReferralCode() {
        try {
            const urlParams = new URLSearchParams(window.location.search);
            const refCodeFromUrl = urlParams.get('ref_code');

            if (refCodeFromUrl) {
                console.log('TRP: ref_code FOUND in URL:', refCodeFromUrl);
                setCookie(cookieName, refCodeFromUrl, cookieExpirationDays);
                finalRefCode = refCodeFromUrl;
                
                // Track landing page visit
                trackLandingPageVisit(refCodeFromUrl);
                
            } else {
                const refCodeFromCookie = getCookie(cookieName);
                if (refCodeFromCookie) {
                    console.log('TRP: ref_code NOT in URL, but FOUND in cookie:', refCodeFromCookie);
                    finalRefCode = refCodeFromCookie;
                } else {
                    console.log('TRP: No ref_code found in URL or cookie.');
                }
            }
            
            return finalRefCode;
            
        } catch (e) {
            console.error("TRP: Error processing URL params or cookies for referral code:", e);
            return null;
        }
    }
    
    /**
     * Update form fields with referral code
     */
    function updateFormFields(referralCode) {
        // Update hidden referral code field
        const referralCodeField = document.getElementById('referral_code_capture');
        if (referralCodeField && referralCode) {
            referralCodeField.value = referralCode;
        }
        
        // Update any other referral code inputs
        const referralInputs = document.querySelectorAll('input[name*="referral"], input[id*="referral"]');
        referralInputs.forEach(input => {
            if (input.type === 'hidden' && referralCode) {
                input.value = referralCode;
            }
        });
    }
    
    /**
     * Update referral notices
     */
    function updateReferralNotices(referralCode) {
        const referralAppliedNotice = document.getElementById('referral-code-applied-notice');
        const manualCodeArea = document.getElementById('manual-referral-code-area');
        
        if (referralCode && referralAppliedNotice) {
            const escapedCode = referralCode.replace(/</g, "&lt;").replace(/>/g, "&gt;");
            updateMainNotice(
                'Referral code <strong>' + escapedCode + '</strong> will be applied.', 
                'success', 
                referralAppliedNotice
            );
            if (manualCodeArea) {
                manualCodeArea.style.display = 'none';
            }
        } else if (referralAppliedNotice) {
            updateMainNotice('No referral code active.', 'info', referralAppliedNotice);
            if (manualCodeArea) {
                manualCodeArea.style.display = 'block';
            }
        }
    }
    
    /**
     * Setup manual referral code entry
     */
    function setupManualCodeEntry() {
        const manualCodeInput = document.getElementById('manual_ref_code_input');
        const applyManualCodeButton = document.getElementById('apply_manual_ref_code_button');
        const manualCodeFeedback = document.getElementById('manual_ref_code_feedback');
        const referralCodeField = document.getElementById('referral_code_capture');
        const referralAppliedNotice = document.getElementById('referral-code-applied-notice');
        
        if (!applyManualCodeButton || !manualCodeInput || !manualCodeFeedback) {
            return; // Elements not found, skip manual code setup
        }
        
        applyManualCodeButton.addEventListener('click', function() {
            const manualCode = manualCodeInput.value.trim();
            manualCodeFeedback.textContent = ''; 

            if (manualCode) {
                console.log('TRP: Applying manually entered code:', manualCode);
                setCookie(cookieName, manualCode, cookieExpirationDays);
                
                if (referralCodeField) {
                    referralCodeField.value = manualCode;
                }
                
                finalRefCode = manualCode;
                
                if (referralAppliedNotice) {
                    const escapedCode = manualCode.replace(/</g, "&lt;").replace(/>/g, "&gt;");
                    updateMainNotice(
                        'Referral code <strong>' + escapedCode + '</strong> applied manually.', 
                        'success', 
                        referralAppliedNotice
                    );
                }
                
                manualCodeInput.value = ''; 
                manualCodeFeedback.textContent = 'Code applied!';
                manualCodeFeedback.style.color = 'var(--trp-success-text, #155724)';
                
                // Hide manual entry area
                const manualCodeArea = document.getElementById('manual-referral-code-area');
                if (manualCodeArea) {
                    manualCodeArea.style.display = 'none';
                }
                
            } else {
                console.log('TRP: Manual code input was empty.');
                manualCodeFeedback.textContent = 'Please enter a code.';
                manualCodeFeedback.style.color = 'var(--trp-error-text, #721c24)';
            }
        });
        
        // Allow Enter key to apply code
        manualCodeInput.addEventListener('keypress', function(e) {
            if (e.key === 'Enter') {
                e.preventDefault();
                applyManualCodeButton.click();
            }
        });
    }
    
    /**
     * Initialize referral tracking
     */
    function initializeReferralTracking() {
        console.log('TRP: Initializing referral tracking');
        
        // Get configuration from localized script
        if (typeof TRP_Config !== 'undefined') {
            cookieName = TRP_Config.cookieName || cookieName;
            cookieExpirationDays = TRP_Config.cookieExpiry || cookieExpirationDays;
            trackApiUrl = TRP_Config.trackApiUrl || trackApiUrl;
            
            console.log('TRP: Configuration loaded:', {
                cookieName: cookieName,
                cookieExpiry: cookieExpirationDays,
                trackApiUrl: trackApiUrl
            });
        } else {
            console.warn('TRP: TRP_Config not found, using default values');
        }
        
        // Process referral code from URL or cookie
        const referralCode = processReferralCode();
        
        // Update form fields and notices
        updateFormFields(referralCode);
        updateReferralNotices(referralCode);
        
        // Setup manual code entry
        setupManualCodeEntry();
        
        console.log('TRP: Referral tracking initialized with code:', referralCode || 'none');
    }
    
    /**
     * Public API for external access
     * Create global object with access to internal functions
     */
    try {
        console.log('TRP: About to create TenantReferralProgram API object...');
        console.log('TRP: Checking required functions...');
        console.log('TRP: setCookie type:', typeof setCookie);
        console.log('TRP: getCookie type:', typeof getCookie);
        console.log('TRP: trackLandingPageVisit type:', typeof trackLandingPageVisit);
        console.log('TRP: updateFormFields type:', typeof updateFormFields);
        console.log('TRP: updateReferralNotices type:', typeof updateReferralNotices);
        
        window.TenantReferralProgram = {
            getCurrentReferralCode: function() {
                return finalRefCode;
            },
            setReferralCode: function(code) {
                if (code) {
                    setCookie(cookieName, code, cookieExpirationDays);
                    finalRefCode = code;
                    updateFormFields(code);
                    updateReferralNotices(code);
                    console.log('TRP: Referral code set programmatically:', code);
                }
            },
            trackLandingPageVisit: trackLandingPageVisit,
            getCookie: getCookie,
            setCookie: setCookie
        };
        
        // Log that the API is available
        console.log('TRP: TenantReferralProgram API created and available');
        console.log('TRP: API object:', window.TenantReferralProgram);
        
    } catch (error) {
        console.error('TRP: Error creating TenantReferralProgram API:', error);
    }
    
    /**
     * Initialize when DOM is ready
     */
    $(document).ready(function() {
        console.log('TRP: JavaScript loaded');
        initializeReferralTracking();
    });
    
    } catch (globalError) {
        console.error('TRP: Global script error:', globalError);
        console.error('TRP: Error stack:', globalError.stack);
    }
    
})(jQuery); 