<?php
/**
 * Form Handler for Tenant Referral Program
 * 
 * Handles referral signup form processing
 * Extracted from template-referral-signup-test.php
 */

if (!defined('ABSPATH')) {
    exit;
}

class Tenant_Referral_Form_Handler {
    
    /**
     * API client instance
     */
    private $api_client;
    
    /**
     * Constructor
     */
    public function __construct() {
        $this->api_client = new Tenant_Referral_API_Client();
    }
    
    /**
     * Validate form data
     * Updated to remove password requirement for friend signups
     * 
     * @param array $post_data Form data
     * @return array Validation result
     */
    public function validate_form_data($post_data) {
        $errors = array();
        
        // Sanitize inputs
        $user_email = isset($post_data['user_email']) ? sanitize_email($post_data['user_email']) : '';
        $display_name = isset($post_data['display_name']) ? sanitize_text_field($post_data['display_name']) : '';
        $phone_number = isset($post_data['phone_number']) ? sanitize_text_field($post_data['phone_number']) : '';
        $post_code = isset($post_data['post_code']) ? sanitize_text_field($post_data['post_code']) : '';
        $address = isset($post_data['address']) ? sanitize_text_field($post_data['address']) : '';
        $consent_terms = isset($post_data['consent_terms']) ? true : false;
        $consent_marketing = isset($post_data['consent_marketing']) ? true : false;
        $referral_code = isset($post_data['referral_code_capture']) ? sanitize_text_field($post_data['referral_code_capture']) : '';
        
        // Also check alternative field name in case of form variations
        if (empty($referral_code)) {
            $referral_code = isset($post_data['trp-referral-code-capture']) ? sanitize_text_field($post_data['trp-referral-code-capture']) : '';
        }
        
        // Validation rules - password requirement removed
        if (empty($user_email) || !is_email($user_email)) {
            $errors[] = 'Please enter a valid email address.';
        }
        // Check if email exists only if we're creating WordPress users (optional)
        // For friend signups, we might want to allow duplicate emails in different contexts
        if (empty($phone_number)) {
            $errors[] = 'Please enter your phone number.';
        }
        if (empty($post_code)) {
            $errors[] = 'Please enter your post code.';
        }
        if (empty($address)) {
            $errors[] = 'Please enter your address.';
        }
        if (!$consent_terms) {
            $errors[] = 'You must agree to the Terms & Conditions to sign up.';
        }
        
        return array(
            'is_valid' => empty($errors),
            'errors' => $errors,
            'sanitized_data' => array(
                'user_email' => $user_email,
                'display_name' => $display_name,
                'phone_number' => $phone_number,
                'post_code' => $post_code,
                'address' => $address,
                'consent_terms' => $consent_terms,
                'consent_marketing' => $consent_marketing,
                'referral_code' => $referral_code
            )
        );
    }
    
    /**
     * Process friend signup without creating WordPress account
     * Friends are tracked as conversions but don't need WordPress access
     * 
     * @param array $validated_data Validated form data
     * @return array Result with success status
     */
    public function process_friend_signup($validated_data) {
        // For friend signups, we don't create WordPress users
        // Instead, we just track the conversion and store the data
        
        // Generate a unique identifier for this friend signup
        $friend_id = 'friend_' . time() . '_' . wp_generate_password(8, false);
        
        // Store friend data in WordPress options or custom table
        $friend_data = array(
            'friend_id' => $friend_id,
            'email' => $validated_data['user_email'],
            'display_name' => $validated_data['display_name'] ?: $validated_data['user_email'],
            'phone_number' => $validated_data['phone_number'],
            'post_code' => $validated_data['post_code'],
            'address' => $validated_data['address'],
            'consent_terms' => $validated_data['consent_terms'],
            'consent_marketing' => $validated_data['consent_marketing'],
            'signup_date' => current_time('mysql'),
            'referral_code' => $validated_data['referral_code']
        );
        
        // Store in WordPress options (for simple implementation)
        // In production, consider using a custom table
        $stored_friends = get_option('trp_friend_signups', array());
        $stored_friends[$friend_id] = $friend_data;
        update_option('trp_friend_signups', $stored_friends);
        
        // Log the friend signup
        error_log("Referral Plugin: Friend signup processed - Email: {$validated_data['user_email']}, Friend ID: $friend_id");
        
        return array(
            'success' => true,
            'friend_id' => $friend_id,
            'message' => 'Friend signup processed successfully'
        );
    }
    
    /**
     * Create WordPress user (kept for backward compatibility or admin users)
     * This method is now optional and used only when WordPress account is actually needed
     * 
     * @param array $validated_data Validated form data
     * @return array Result with success status
     */
    public function create_wordpress_user($validated_data) {
        // Generate a random password since user didn't provide one
        $generated_password = wp_generate_password(12, true);
        
        $user_data = array(
            'user_login'   => $validated_data['user_email'],
            'user_email'   => $validated_data['user_email'],
            'user_pass'    => $generated_password,
            'display_name' => $validated_data['display_name'] ?: $validated_data['user_email'],
            'role'         => 'subscriber',
        );
        
        $user_id = wp_insert_user($user_data);
        
        if (is_wp_error($user_id)) {
            return array(
                'success' => false,
                'error' => $user_id->get_error_message()
            );
        }
        
        // Add user metadata
        $this->add_user_metadata($user_id, $validated_data);
        
        // Send password reset email so user can set their own password if they want to log in
        wp_new_user_notification($user_id, null, 'user');
        
        return array(
            'success' => true,
            'user_id' => $user_id,
            'password_sent' => true
        );
    }
    
    /**
     * Add user metadata
     * Extracted from template-referral-signup-test.php
     * 
     * @param int $user_id WordPress user ID
     * @param array $data Validated form data
     */
    private function add_user_metadata($user_id, $data) {
        update_user_meta($user_id, 'billing_phone', $data['phone_number']);
        update_user_meta($user_id, 'billing_postcode', $data['post_code']);
        update_user_meta($user_id, 'billing_address_1', $data['address']);
        update_user_meta($user_id, 'terms_agreed', $data['consent_terms'] ? 'yes' : 'no');
        update_user_meta($user_id, 'marketing_consent', $data['consent_marketing'] ? 'yes' : 'no');
    }
    
    /**
     * Track referral conversion
     * Updated to work with friend signups that don't create WordPress users
     * 
     * @param string $customer_id Customer identifier (friend_id or user_id)
     * @param string $referral_code Referral code
     * @param array $customer_data Customer data
     * @return array Result with success status
     */
    public function track_referral_conversion($customer_id, $referral_code, $customer_data) {
        if (empty($referral_code)) {
            return array(
                'success' => false,
                'message' => 'No referral code provided'
            );
        }
        
        // Build payload - updated structure for friend signups
        $payload = array(
            'referralCode'     => $referral_code,
            'event_type'       => 'CONVERSION',
            'newCustomerEmail' => $customer_data['user_email'],
            'newCustomerName'  => $customer_data['display_name'] ?: $customer_data['user_email'],
            'newCustomerIdWp'  => (string)$customer_id, // WordPress customer ID (friend_id or user_id)
            'newCustomerPhone' => $customer_data['phone_number'],
            'newCustomerPostCode' => $customer_data['post_code'],
            'newCustomerAddress' => $customer_data['address']
        );
        
        // Log the payload
        error_log("Referral Plugin: New customer {$customer_data['user_email']} (ID: $customer_id) registered with referral code: $referral_code");
        error_log("WP Payload to Next.js API: " . print_r($payload, true));
        
        // Get track API URL
        $track_api_url = $this->get_track_api_url();
        
        // Prepare request headers
        $request_headers = array('Content-Type' => 'application/json');
        if (strpos($track_api_url, 'ngrok-free.app') !== false || strpos($track_api_url, 'ngrok.io') !== false) {
            $request_headers['ngrok-skip-browser-warning'] = 'true';
        }
        
        $args = array(
            'body'        => json_encode($payload),
            'headers'     => $request_headers,
            'method'      => 'POST',
            'data_format' => 'body',
            'timeout'     => 5,
        );
        
        error_log("Referral Plugin: Attempting to POST to URL: " . $track_api_url . " with payload: " . json_encode($payload));
        
        $response = wp_remote_post($track_api_url, $args);
        
        if (is_wp_error($response)) {
            $error_message = $response->get_error_message();
            error_log("Referral Plugin (Track Error): Failed for code " . $referral_code . ". Error: " . $error_message);
            return array(
                'success' => false,
                'message' => 'Referral tracking API call failed: ' . $error_message
            );
        }
        
        $status_code = wp_remote_retrieve_response_code($response);
        $body = wp_remote_retrieve_body($response);
        
        if ($status_code >= 200 && $status_code < 300) {
            error_log("Referral Plugin (Track Success): Conversion for code " . $referral_code . " tracked. Response: " . $body);
            return array(
                'success' => true,
                'message' => 'Referral tracking successful.'
            );
        } else {
            error_log("Referral Plugin (Track API Error): Code " . $referral_code . ". Status: " . $status_code . ". Body: " . $body);
            return array(
                'success' => false,
                'message' => 'Referral tracking API returned status ' . $status_code . ". Details: " . $body
            );
        }
    }
    
    /**
     * Get track API URL
     * Uses same logic as template for determining API URL
     * 
     * @return string Track API URL
     */
    private function get_track_api_url() {
        $default_api_base_url = 'http://localhost:3000';
        $api_base_url_from_option = get_option('trp_base_url', $default_api_base_url);
        return rtrim($api_base_url_from_option, '/') . '/api/referrals/track';
    }
} 