<?php
/**
 * Assets Management for Tenant Referral Program
 * 
 * Handles CSS and JavaScript enqueuing with proper dependency management
 * and configuration passing to JavaScript
 */

if (!defined('ABSPATH')) {
    exit;
}

class Tenant_Referral_Assets {
    
    private $plugin_name;
    private $version;
    private $has_shortcode = false;
    
    /**
     * Constructor
     */
    public function __construct($plugin_name, $version) {
        $this->plugin_name = $plugin_name;
        $this->version = $version;
        
        // Enqueue assets for frontend
        add_action('wp_enqueue_scripts', array($this, 'enqueue_public_assets'));
        add_action('wp_footer', array($this, 'check_for_shortcodes'));
        
        // Also enqueue assets for admin when needed (for shortcode testing)
        add_action('admin_enqueue_scripts', array($this, 'enqueue_admin_assets_conditional'));
    }
    
    /**
     * Check if any referral shortcodes are present on the page
     */
    public function check_for_shortcodes() {
        global $post;
        
        if (is_a($post, 'WP_Post')) {
            // Check for our shortcodes in post content
            if (has_shortcode($post->post_content, 'tenant_referral_widget') ||
                has_shortcode($post->post_content, 'inalan_referral_widget') ||
                has_shortcode($post->post_content, 'tenant_referral_signup')) {
                $this->has_shortcode = true;
            }
        }
        
        // Also check if shortcodes are used in widgets or other areas
        if (is_active_widget(false, false, 'text') || 
            is_active_widget(false, false, 'custom_html')) {
            // For now, assume shortcodes might be in widgets
            // In a more sophisticated implementation, we'd parse widget content
            $this->has_shortcode = true;
        }
    }
    
    /**
     * Enqueue public-facing assets
     */
    public function enqueue_public_assets() {
        // Debug: Check if TRP_PLUGIN_URL is defined
        if (!defined('TRP_PLUGIN_URL')) {
            error_log('TRP: ERROR - TRP_PLUGIN_URL is not defined!');
            return;
        }
        error_log('TRP: TRP_PLUGIN_URL is: ' . TRP_PLUGIN_URL);
        // Always enqueue CSS (it's lightweight)
        wp_enqueue_style(
            $this->plugin_name . '-public',
            TRP_PLUGIN_URL . 'assets/css/public-styles.css',
            array(),
            $this->version,
            'all'
        );
        
        // Enqueue JavaScript with jQuery dependency
        wp_enqueue_script(
            $this->plugin_name . '-public',
            TRP_PLUGIN_URL . 'assets/js/public-scripts.js',
            array('jquery'),
            $this->version . '-' . time(), // Force cache bust
            true // Load in footer
        );
        
        // Debug: Log that we're trying to enqueue the script
        error_log('TRP: Enqueuing script: ' . TRP_PLUGIN_URL . 'assets/js/public-scripts.js');
        
        // Localize script with configuration data
        $this->localize_script_config();
    }
    
    /**
     * Pass configuration data to JavaScript
     */
    private function localize_script_config() {
        // Get plugin settings
        $base_url = get_option('trp_base_url', '');
        $cookie_name = get_option('trp_cookie_name', TRP_COOKIE_NAME);
        $cookie_expiry = get_option('trp_cookie_expiry_days', TRP_COOKIE_EXPIRY_DAYS);
        
        // Build API URLs
        $track_api_url = '';
        if (!empty($base_url)) {
            $track_api_url = rtrim($base_url, '/') . TRP_API_ENDPOINT;
        }
        
        // Prepare configuration array
        $config = array(
            'cookieName' => $cookie_name,
            'cookieExpiry' => intval($cookie_expiry),
            'trackApiUrl' => $track_api_url,
            'baseUrl' => $base_url,
            'widgetEndpoint' => TRP_WIDGET_ENDPOINT,
            'apiEndpoint' => TRP_API_ENDPOINT,
            'ajaxUrl' => admin_url('admin-ajax.php'),
            'nonce' => wp_create_nonce('trp_ajax_nonce'),
            'debug' => defined('WP_DEBUG') && WP_DEBUG,
            'version' => $this->version
        );
        
        // Add legacy support
        $legacy_url = get_option('inalan_referral_widget_url', '');
        if (!empty($legacy_url) && empty($base_url)) {
            $config['trackApiUrl'] = rtrim($legacy_url, '/') . TRP_API_ENDPOINT;
            $config['baseUrl'] = $legacy_url;
        }
        
        // Localize the script
        wp_localize_script(
            $this->plugin_name . '-public',
            'TRP_Config',
            $config
        );
        
        // Add inline script for debugging if WP_DEBUG is enabled
        if (defined('WP_DEBUG') && WP_DEBUG) {
            $debug_script = "
                console.log('TRP: Configuration loaded', TRP_Config);
                console.log('TRP: Plugin version " . $this->version . "');
                console.log('TRP: JavaScript URL should be: " . TRP_PLUGIN_URL . "assets/js/public-scripts.js');
            ";
            wp_add_inline_script($this->plugin_name . '-public', $debug_script);
        }
    }
    
    /**
     * Enqueue admin assets conditionally (for shortcode testing in admin)
     */
    public function enqueue_admin_assets_conditional($hook) {
        // Check if we're on a page that might have shortcodes
        global $post;
        
        $should_load = false;
        
        // Load on our settings page
        if ($hook === 'settings_page_tenant-referral-program') {
            $should_load = true;
        }
        
        // Load on post edit pages that might contain shortcodes
        if (in_array($hook, array('post.php', 'post-new.php', 'page.php', 'page-new.php'))) {
            $should_load = true;
        }
        
        // Load if current page content contains our shortcodes
        if (is_a($post, 'WP_Post')) {
            if (has_shortcode($post->post_content, 'tenant_referral_test') ||
                has_shortcode($post->post_content, 'tenant_referral_widget') ||
                has_shortcode($post->post_content, 'tenant_referral_signup')) {
                $should_load = true;
            }
        }
        
        if ($should_load) {
            // Enqueue the same public assets for admin context
            $this->enqueue_public_assets();
        }
    }
    
    /**
     * Enqueue admin assets (for settings page)
     */
    public function enqueue_admin_assets($hook) {
        // Only load on our settings page
        if ($hook !== 'settings_page_tenant-referral-program') {
            return;
        }
        
        wp_enqueue_style(
            $this->plugin_name . '-admin',
            TRP_PLUGIN_URL . 'assets/css/admin-styles.css',
            array(),
            $this->version,
            'all'
        );
        
        wp_enqueue_script(
            $this->plugin_name . '-admin',
            TRP_PLUGIN_URL . 'assets/js/admin-scripts.js',
            array('jquery'),
            $this->version,
            true
        );
    }
    
    /**
     * Check if shortcodes are present (for conditional loading)
     */
    public function has_referral_shortcodes() {
        return $this->has_shortcode;
    }
    
    /**
     * Force enqueue assets (for programmatic use)
     */
    public function force_enqueue_assets() {
        $this->has_shortcode = true;
        $this->enqueue_public_assets();
    }
    
    /**
     * Get asset URLs for external use
     */
    public function get_asset_urls() {
        return array(
            'css' => TRP_PLUGIN_URL . 'assets/css/public-styles.css',
            'js' => TRP_PLUGIN_URL . 'assets/js/public-scripts.js',
            'version' => $this->version
        );
    }
} 